/**
 * Root representation of a Markdown document.
 */
class Document {
	// Create an empty document.
	init(ElementSeq elements) {
		init { elements = elements; }
	}

	// Elements in the document.
	ElementSeq elements;

	// Visit the entire structure using a `Visitor`. The visitor allows modifying the elements as
	// well.
	void visit(Visitor v) {
		elements = elements.visit(v);
	}

	// Create a string representation of the markdown document.
	void toS(StrBuf to) : override {
		elements.toS(to);
	}

	// Create a HTML representation of the document.
	Html toHtml() {
		Html out;
		elements.toHtml(out);
		out;
	}
}

/**
 * Collection of elements.
 */
class ElementSeq {
	// Elements.
	Element[] elements;

	// Create.
	init() {
		init {}
	}

	// Add an element.
	void add(Element e) {
		if (e as PackedElements)
			add(e.elements);
		else
			elements << e;
	}

	// Add many elements.
	void add(ElementSeq e) {
		elements.append(e.elements);
	}

	// Remove an element.
	void remove(Nat id) {
		elements.remove(id);
	}

	// Visit the entire structure.
	ElementSeq visit(Visitor v) {
		Element[] newElems;
		for (x in elements) {
			var newElem = x.visit(v);
			if (newElem as PackedElements) {
				newElems.append(newElem.elements.elements);
			} else {
				newElems << newElem;
			}
		}

		elements = newElems;

		return v.visit(this);
	}

	// To string.
	void toS(StrBuf to) : override {
		to << join(elements, "\n\n");
	}

	// To HTML.
	void toHtml(Html to) {
		for (i, element in elements) {
			if (i > 0)
				to.html("\n");
			element.toHtml(to);
		}
	}

	// To HTML, ignore first paragraph.
	void toHtmlInline(Html to) {
		for (i, element in elements) {
			if (i == 0) {
				if (element as Paragraph) {
					element.text.toHtml(to);
					continue;
				}
			} else {
				to.html("\n");
			}

			element.toHtml(to);
		}
	}
}


/**
 * Element in a document.
 */
class Element {
	// Visit the element. The function is expected to run `v.visit(this)` as the last step. The
	// returned value is the element that replaces this element after the visiting operation. If
	// nothing is changed, then the object itself is returned.
	Element visit(Visitor v) : abstract;

	// Create an HTML representation of this object. The `Html` object is essentially a string
	// stream, but with logic for escaping text in the context of HTML.
	void toHtml(Html to) : abstract;
}


/**
 * Heading.
 */
class Heading extends Element {
	// Heading level.
	Nat level;

	// Text.
	FormattedText text;

	// Create.
	init(Nat level, FormattedText text) {
		init { level = level; text = text; }
	}

	// Visit.
	Element visit(Visitor v) {
		text = text.visit(v);
		return v.visit(this);
	}

	// Output.
	protected void toS(StrBuf to) : override {
		to << "#"*level << " " << text;
	}

	// To HTML.
	void toHtml(Html to) : override {
		to.html("<h${level}>");
		text.toHtml(to);
		to.html("</h${level}>");
	}
}


/**
 * A paragraph of text.
 */
class Paragraph extends Element {
	// The text in the paragraph.
	FormattedText text;

	// Create.
	init(FormattedText text) {
		init { text = text; }
	}

	// Visit.
	Element visit(Visitor v) {
		text = text.visit(v);
		return v.visit(this);
	}

	// Output.
	protected void toS(StrBuf to) : override {
		to << text;
	}

	// To HTML.
	void toHtml(Html to) : override {
		to.html("<p>\n");
		to.indent();
		text.toHtml(to);
		to.dedent();
		to.html("\n</p>");
	}
}


/**
 * A generic list.
 */
class List extends Element {
	// Ordered?
	Bool ordered;

	// Contents of the list.
	ElementSeq[] items;

	// Create.
	init(Bool ordered) {
		init { ordered = ordered; }
	}

	// Add an element. Content can be added later.
	void add(ElementSeq item) {
		items << item;
	}

	// Visit.
	Element visit(Visitor v) {
		for (i, x in items)
			items[i] = x.visit(v);
		return v.visit(this);
	}

	// To string.
	protected void toS(StrBuf to) : override {
		if (ordered) {
			Nat width = items.count.toS.count;

			to.indentBy(" "*(width + 2));
			for (i, e in items) {
				if (i > 0)
					to << "\n\n";
				to << width(width) << fill('0') << (i + 1) << ". ";
				{
					Indent z(to);
					e.toS(to);
				}
			}
		} else {
			to.indentBy("  ");
			for (i, e in items) {
				if (i > 0)
					to << "\n\n";
				to << "- ";
				{
					Indent z(to);
					e.toS(to);
				}
			}
		}
	}

	// To HTML.
	void toHtml(Html to) : override {
		if (ordered)
			to.html("<ol>\n");
		else
			to.html("<ul>\n");
		to.indent();
		for (e in items) {
			to.html("<li>\n");
			to.indent();
			e.toHtmlInline(to);
			to.html("\n");
			to.dedent();
			to.html("</li>\n");
		}
		to.dedent();
		if (ordered)
			to.html("</ol>");
		else
			to.html("</ul>");
	}
}


/**
 * Code.
 */
class CodeBlock extends Element {
	// Language of the code, if specified.
	Str language;

	// Lines of code in the code block.
	Str[] code;

	// Create.
	init(Str language) {
		init { language = language; }
	}

	// Add a line of code.
	void addLine(Str line) {
		code << line;
	}

	// Visit.
	Element visit(Visitor v) {
		return v.visit(this);
	}

	// To string.
	protected void toS(StrBuf to) : override {
		to << "```" << language << "\n";
		to << join(code, "\n") << "\n";
		to << "```";
	}

	// To HTML.
	void toHtml(Html to) : override {
		to.html("<pre>");
		to.pauseIndent();
		for (i, l in code) {
			if (i > 0)
				to.text("\n");
			to.text(l);
		}
		to.restoreIndent();
		to.html("</pre>");
	}
}


/**
 * Element that contains a sequence of other elements. Useful when replacing a single element with
 * multiple elements in a visitor.
 */
class PackedElements extends Element {
	// Elements.
	ElementSeq elements;

	// Create.
	init() {
		init {}
	}

	// Add an element.
	void add(Element elem) {
		elements.add(elem);
	}

	// Add a sequence of elements.
	void add(ElementSeq seq) {
		elements.add(seq);
	}

	// Visit elements.
	Element visit(Visitor v) : override {
		elements = elements.visit(v);
		return v.visit(this);
	}

	// To string.
	protected void toS(StrBuf to) : override {
		to << elements;
	}

	// To HTML.
	void toHtml(Html to) : override {
		elements.toHtml(to);
	}
}


/**
 * Custom HTML element.
 *
 * Used to make it convenient for other tools to generate custom HTML in the document. Not currently
 * created by the parser when parsing markdown.
 */
class CustomHtml extends Element {
	// HTML code.
	Str code;

	// Skip indentation?
	Bool skipIndent;

	// Create.
	init(Str code) {
		init { code = code; }
	}

	// Create, specify skip indent.
	init(Str code, Bool skipIndent) {
		init { code = code; skipIndent = skipIndent; }
	}

	// Visit.
	Element visit(Visitor v) : override {
		return v.visit(this);
	}

	// To string.
	protected void toS(StrBuf to) : override {
		to << "!! begin custom html !!\n";
		to << code;
		to << "\n!! end custom html !!";
	}

	// Output HTML.
	void toHtml(Html to) : override {
		if (skipIndent)
			to.pauseIndent();
		to.html(code);
		if (skipIndent)
			to.restoreIndent();
	}
}
