/*
 *  @(#)Log4jLog.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.autodoc.v1.log4j;


import net.sourceforge.groboutils.autodoc.v1.AutoDocLog;

import org.apache.log4j.Logger;
import org.apache.log4j.Priority;
 

/**
 * An interface for logging.  This allows for an abstraction between the
 * owning class and any underlying logging mechanism desired.
 * <P>
 * The actual meaning of the logging levels is implementation independent.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:52:12 $
 * @since      March 16, 2002
 */
public class Log4jLog implements AutoDocLog
{
    private Logger log;
    
    
    /**
     * Standard constructor - uses the Logger based on the given class.
     *
     * @param owner the owning class to assign log responsibility to.
     *      Must not be <tt>null</tt>.
     * @exception IllegalArgumentException if <tt>owner</tt> is <tt>null</tt>.
     */
    public Log4jLog( Class owner )
    {
        if (owner == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        setLog( Logger.getLogger( owner ) );
    }
    
    
    /**
     * Create a log interface based on the given Apache Logger.
     *
     * @param log the underlying log instance to use for logging.  Must not
     *      be <tt>null</tt>.
     * @exception IllegalArgumentException if <tt>owner</tt> is <tt>null</tt>.
     */
    public Log4jLog( Logger log )
    {
        setLog( log );
    }
    
    
    public void debug( Object message )
    {
        this.log.debug( message );
    }

    public void debug( Object message[] )
    {
        if (this.log.isDebugEnabled())
        {
            this.log.debug( concatMessage( message ) );
        }
    }
    
    public void debug( Object message, Throwable error )
    {
        this.log.debug( message, error );
    }

    public void debug( Object message[], Throwable error )
    {
        if (this.log.isDebugEnabled())
        {
            this.log.debug( concatMessage( message ), error );
        }
    }
    
    public void info( Object message )
    {
        this.log.info( message );
    }

    public void info( Object message[] )
    {
        if (this.log.isInfoEnabled())
        {
            this.log.info( concatMessage( message ) );
        }
    }
    
    public void info( Object message, Throwable error )
    {
        this.log.info( message, error );
    }

    public void info( Object message[], Throwable error )
    {
        if (this.log.isInfoEnabled())
        {
            Object msg = concatMessage( message );
            this.log.info( msg, error );
        }
    }
    
    public void warn( Object message )
    {
        this.log.warn( message );
    }

    public void warn( Object message[] )
    {
        if (this.log.isEnabledFor( Priority.WARN ))
        {
            this.log.warn( concatMessage( message ) );
        }
    }
    
    public void warn( Object message, Throwable error )
    {
        this.log.warn( message, error );
    }
    
    public void warn( Object message[], Throwable error )
    {
        if (this.log.isEnabledFor( Priority.WARN ))
        {
            this.log.warn( concatMessage( message ), error );
        }
    }
    
    
    /**
     * Concatenate the given array into a single string.  The returned object
     * will be of type <tt>java.lang.StringBuffer</tt> or
     * <tt>java.lang.String</tt>; use <tt>toString()</tt> on the resulting
     * object.
     *
     * @param o an array (possibly null) of objects (possibly null) to
     *      concatenate their <tt>toString()</tt> results together.
     * @return the concatenated message.
     * @see java.lang.StringBuffer
     */
    protected Object concatMessage( Object o[] )
    {
        if (o == null)
        {
            return "null";
        }
        if (o.length <= 0)
        {
            return "";
        }
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < o.length; ++i)
        {
            sb.append( o[i] );
        }
        return sb;
    }
    
    
    /**
     * Sets the internal log instance.
     *
     * @param log the log to set.  It must not be <tt>null</tt>.
     * @exception IllegalArgumentException if <tt>log</tt> is <tt>null</tt>.
     */
    protected void setLog( Logger log )
    {
        if (log == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        this.log = log;
    }
}

