/*
 * jclassinfo
 * Copyright (C) 2003  Nicos Panayides
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Nicos Panayides
 * anarxia@gmx.net
 *
 * $Id: class.c,v 1.5 2004/04/14 02:18:33 anarxia Exp $
 */

#include "config.h"
#include <stdlib.h>
#include <jclass/jclass.h>
#include "common.h"
#include "attributes.h"
#include "attributes_xml.h"
#include "constant_pool.h"
#include "field.h"

void print_class(JavaClass* curr_class, int do_flag, JCVisibility visibility)
{
	char* this_class;
	char* this_package;
	char* class_name;
	char **interfaces;
	char* access_flag;
	int count, i;

	this_class = jclass_class_get_class_name(curr_class);
	this_package = jclass_get_package_from_class_name(this_class);
	
	if(do_flag & GENERAL_INFO)
	{
		access_flag = jclass_access_flag_to_string(curr_class->access_flags, 1);
			
		if(do_flag & SECTION_TITLES)
			puts("[CLASS INFORMATION]");

		printf("%s %s", access_flag, this_class);
				
		free(access_flag);
	
		class_name = jclass_class_get_super_class_name(curr_class);
		if (class_name != NULL)
		{
			printf(" extends %s", class_name);	
			free(class_name);
		}
		
		interfaces = jclass_class_get_interfaces(curr_class);
		
		if (interfaces)
		{	
			printf(" implements %s", interfaces[0]);
			free(interfaces[0]);
			for(i=1; interfaces[i]; i++)
			{
				printf(", %s", interfaces[i]);
				free(interfaces[i]);
			}
			free(interfaces);
		}
		puts("");
			
		class_name = jclass_class_get_sourcefile_name(curr_class);
			
		if(class_name != NULL)
		{
			printf("Compiled from %s\n", class_name);
			free(class_name);
		}
			
		printf("Requires: Java VM %s or higher\n", jclass_class_get_vm_spec(curr_class));
	}

	
	if(do_flag & CONSTANT_POOL)
	{
		if(do_flag & SECTION_TITLES)
		{
			puts("[CONSTANT POOL]");
			printf("Size: %d\n", curr_class->constant_pool->count);
		}
		cp_print(curr_class->constant_pool);
	}
		
	if(do_flag & FIELDS)
	{
		if(do_flag & SECTION_TITLES)
		{
			puts("[FIELDS]");
			if(curr_class->fields_count == 0)
				puts("none");
		}
			
		for(count = 0; count < curr_class->fields_count; count++)
		{
			if(jclass_field_is_visible(&curr_class->fields[count], curr_class->constant_pool, visibility))
					field_print(&curr_class->fields[count], curr_class->constant_pool, do_flag);
		}
	}

	if(do_flag & METHODS)
	{
		if(do_flag & SECTION_TITLES)
		{
			puts("[METHODS]");
			if(!curr_class->methods_count)
				puts("none");
		}
			
		for(count=0; count < curr_class->methods_count; count++)
		{
			if(jclass_field_is_visible(&curr_class->methods[count], curr_class->constant_pool, visibility))
					field_print(&curr_class->methods[count], curr_class->constant_pool, do_flag);
		}
			
	}

	if(do_flag & ATTRIBUTES)
	{	
		if(do_flag & SECTION_TITLES) 
		{
			puts("[ATTRIBUTES]");
			if(!curr_class->attributes_count)
				puts("none");
		}						
		for(count = 0; count < curr_class->attributes_count; count++)
		{
			attribute_container_print(&curr_class->attributes[count], INT_IS_INT, 0, curr_class->constant_pool);
			puts("");
		}
	}
	
	free(this_class);
	if(this_package != NULL)
		free(this_package);
}

void print_class_xml(JavaClass* curr_class, int do_flag, JCVisibility visibility)
{
	char* this_class;
	char* this_package;
	char* class_name;
	char **interfaces;
	char* access_flag;
	int count, i;

	this_class = jclass_class_get_class_name(curr_class);
	this_package = jclass_get_package_from_class_name(this_class);
	
	printf("<class name=\"%s\"", this_class);
	if(this_package != NULL)
		printf(" package=\"%s\"", this_package);
	
	if(do_flag & GENERAL_INFO)
	{
		access_flag = jclass_access_flag_to_string(curr_class->access_flags, 1);
			
		printf(" access=\"%s\"", access_flag);		
		free(access_flag);
	
		class_name = jclass_class_get_super_class_name(curr_class);
		if (class_name != NULL)
		{
			printf(" extends=\"%s\"", class_name);
			free(class_name);
		}
		
			
		class_name = jclass_class_get_sourcefile_name(curr_class);
			
		if(class_name != NULL)
		{
			printf(" sourcefile=\"%s\"", class_name);
			free(class_name);
		}
			
		printf(" vm=\"%s\"", jclass_class_get_vm_spec(curr_class));
	}

	printf(">\n");

	if (do_flag & GENERAL_INFO)
	{
		interfaces = jclass_class_get_interfaces(curr_class);
		if (interfaces)
		{
			for(i=0; interfaces[i]; i++)
			{
				printf("<implements name=\"%s\"/>\n", interfaces[i]);
				free(interfaces[i]);
			}
			free(interfaces);
		}
	}
	
	if(do_flag & CONSTANT_POOL)
		cp_print_xml(curr_class->constant_pool);
		
	if(do_flag & FIELDS)
	{		
		for(count = 0; count < curr_class->fields_count; count++)
		{
			if(jclass_field_is_visible(&curr_class->fields[count], curr_class->constant_pool, visibility))
					field_print_xml(&curr_class->fields[count], curr_class->constant_pool, 0, 0);
		}
	}

	if(do_flag & METHODS)
	{
		for(count=0; count < curr_class->methods_count; count++)
		{
			if(jclass_field_is_visible(&curr_class->methods[count], curr_class->constant_pool, visibility))
				field_print_xml(&curr_class->methods[count], curr_class->constant_pool, do_flag, 1);
		}
			
	}

	if(do_flag & ATTRIBUTES)
	{	
		for(count = 0; count < curr_class->attributes_count; count++)
			attribute_container_print_xml(&curr_class->attributes[count], INT_IS_INT, 0, curr_class->constant_pool);
	}
	
	puts("</class>");
	
	free(this_class);
	if(this_package != NULL)
		free(this_package);
}
