use core::pin::Pin;
use core::task::{Context, Poll};
use futures_core::future::{FusedFuture, Future};
use futures_core::stream::{FusedStream, Stream};
#[cfg(feature = "sink")]
use futures_sink::Sink;
use pin_project::{pin_project, project};

/// Combines two different futures, streams, or sinks having the same associated types into a single
/// type.
#[pin_project]
#[derive(Debug, Clone)]
pub enum Either<A, B> {
    /// First branch of the type
    Left(#[pin] A),
    /// Second branch of the type
    Right(#[pin] B),
}

impl<A, B, T> Either<(T, A), (T, B)> {
    /// Factor out a homogeneous type from an either of pairs.
    ///
    /// Here, the homogeneous type is the first element of the pairs.
    pub fn factor_first(self) -> (T, Either<A, B>) {
        match self {
            Either::Left((x, a)) => (x, Either::Left(a)),
            Either::Right((x, b)) => (x, Either::Right(b)),
        }
    }
}

impl<A, B, T> Either<(A, T), (B, T)> {
    /// Factor out a homogeneous type from an either of pairs.
    ///
    /// Here, the homogeneous type is the second element of the pairs.
    pub fn factor_second(self) -> (Either<A, B>, T) {
        match self {
            Either::Left((a, x)) => (Either::Left(a), x),
            Either::Right((b, x)) => (Either::Right(b), x),
        }
    }
}

impl<T> Either<T, T> {
    /// Extract the value of an either over two equivalent types.
    pub fn into_inner(self) -> T {
        match self {
            Either::Left(x) => x,
            Either::Right(x) => x,
        }
    }
}

impl<A, B> Future for Either<A, B>
where
    A: Future,
    B: Future<Output = A::Output>,
{
    type Output = A::Output;

    #[project]
    fn poll(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<A::Output> {
        #[project]
        match self.project() {
            Either::Left(x) => x.poll(cx),
            Either::Right(x) => x.poll(cx),
        }
    }
}

impl<A, B> FusedFuture for Either<A, B>
where
    A: FusedFuture,
    B: FusedFuture<Output = A::Output>,
{
    fn is_terminated(&self) -> bool {
        match self {
            Either::Left(x) => x.is_terminated(),
            Either::Right(x) => x.is_terminated(),
        }
    }
}

impl<A, B> Stream for Either<A, B>
where
    A: Stream,
    B: Stream<Item = A::Item>,
{
    type Item = A::Item;

    #[project]
    fn poll_next(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Option<A::Item>> {
        #[project]
        match self.project() {
            Either::Left(x) => x.poll_next(cx),
            Either::Right(x) => x.poll_next(cx),
        }
    }
}

impl<A, B> FusedStream for Either<A, B>
where
    A: FusedStream,
    B: FusedStream<Item = A::Item>,
{
    fn is_terminated(&self) -> bool {
        match self {
            Either::Left(x) => x.is_terminated(),
            Either::Right(x) => x.is_terminated(),
        }
    }
}

#[cfg(feature = "sink")]
impl<A, B, Item> Sink<Item> for Either<A, B>
where
    A: Sink<Item>,
    B: Sink<Item, Error = A::Error>,
{
    type Error = A::Error;

    #[project]
    fn poll_ready(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Result<(), Self::Error>> {
        #[project]
        match self.project() {
            Either::Left(x) => x.poll_ready(cx),
            Either::Right(x) => x.poll_ready(cx),
        }
    }

    #[project]
    fn start_send(self: Pin<&mut Self>, item: Item) -> Result<(), Self::Error> {
        #[project]
        match self.project() {
            Either::Left(x) => x.start_send(item),
            Either::Right(x) => x.start_send(item),
        }
    }

    #[project]
    fn poll_flush(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Result<(), Self::Error>> {
        #[project]
        match self.project() {
            Either::Left(x) => x.poll_flush(cx),
            Either::Right(x) => x.poll_flush(cx),
        }
    }

    #[project]
    fn poll_close(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Result<(), Self::Error>> {
        #[project]
        match self.project() {
            Either::Left(x) => x.poll_close(cx),
            Either::Right(x) => x.poll_close(cx),
        }
    }
}

#[cfg(feature = "io")]
#[cfg(feature = "std")]
mod if_std {
    use super::*;

    use core::pin::Pin;
    use core::task::{Context, Poll};
    #[cfg(feature = "read-initializer")]
    use futures_io::Initializer;
    use futures_io::{
        AsyncBufRead, AsyncRead, AsyncSeek, AsyncWrite, IoSlice, IoSliceMut, Result, SeekFrom,
    };

    impl<A, B> AsyncRead for Either<A, B>
    where
        A: AsyncRead,
        B: AsyncRead,
    {
        #[cfg(feature = "read-initializer")]
        unsafe fn initializer(&self) -> Initializer {
            match self {
                Either::Left(x) => x.initializer(),
                Either::Right(x) => x.initializer(),
            }
        }

        #[project]
        fn poll_read(
            self: Pin<&mut Self>,
            cx: &mut Context<'_>,
            buf: &mut [u8],
        ) -> Poll<Result<usize>> {
            #[project]
            match self.project() {
                Either::Left(x) => x.poll_read(cx, buf),
                Either::Right(x) => x.poll_read(cx, buf),
            }
        }

        #[project]
        fn poll_read_vectored(
            self: Pin<&mut Self>,
            cx: &mut Context<'_>,
            bufs: &mut [IoSliceMut<'_>],
        ) -> Poll<Result<usize>> {
            #[project]
            match self.project() {
                Either::Left(x) => x.poll_read_vectored(cx, bufs),
                Either::Right(x) => x.poll_read_vectored(cx, bufs),
            }
        }
    }

    impl<A, B> AsyncWrite for Either<A, B>
    where
        A: AsyncWrite,
        B: AsyncWrite,
    {
        #[project]
        fn poll_write(
            self: Pin<&mut Self>,
            cx: &mut Context<'_>,
            buf: &[u8],
        ) -> Poll<Result<usize>> {
            #[project]
            match self.project() {
                Either::Left(x) => x.poll_write(cx, buf),
                Either::Right(x) => x.poll_write(cx, buf),
            }
        }

        #[project]
        fn poll_write_vectored(
            self: Pin<&mut Self>,
            cx: &mut Context<'_>,
            bufs: &[IoSlice<'_>],
        ) -> Poll<Result<usize>> {
            #[project]
            match self.project() {
                Either::Left(x) => x.poll_write_vectored(cx, bufs),
                Either::Right(x) => x.poll_write_vectored(cx, bufs),
            }
        }

        #[project]
        fn poll_flush(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Result<()>> {
            #[project]
            match self.project() {
                Either::Left(x) => x.poll_flush(cx),
                Either::Right(x) => x.poll_flush(cx),
            }
        }

        #[project]
        fn poll_close(self: Pin<&mut Self>, cx: &mut Context<'_>) -> Poll<Result<()>> {
            #[project]
            match self.project() {
                Either::Left(x) => x.poll_close(cx),
                Either::Right(x) => x.poll_close(cx),
            }
        }
    }

    impl<A, B> AsyncSeek for Either<A, B>
    where
        A: AsyncSeek,
        B: AsyncSeek,
    {
        #[project]
        fn poll_seek(
            self: Pin<&mut Self>,
            cx: &mut Context<'_>,
            pos: SeekFrom,
        ) -> Poll<Result<u64>> {
            #[project]
            match self.project() {
                Either::Left(x) => x.poll_seek(cx, pos),
                Either::Right(x) => x.poll_seek(cx, pos),
            }
        }
    }

    impl<A, B> AsyncBufRead for Either<A, B>
    where
        A: AsyncBufRead,
        B: AsyncBufRead,
    {
        #[project]
        fn poll_fill_buf(
            self: Pin<&mut Self>,
            cx: &mut Context<'_>,
        ) -> Poll<Result<&[u8]>> {
            #[project]
            match self.project() {
                Either::Left(x) => x.poll_fill_buf(cx),
                Either::Right(x) => x.poll_fill_buf(cx),
            }
        }

        #[project]
        fn consume(self: Pin<&mut Self>, amt: usize) {
            #[project]
            match self.project() {
                Either::Left(x) => x.consume(amt),
                Either::Right(x) => x.consume(amt),
            }
        }
    }
}
